/**
 * SPDX-FileCopyrightText: 2024-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#ifndef AURORA_MDM_MANAGEDCONFIGURATION_H
#define AURORA_MDM_MANAGEDCONFIGURATION_H

#include <functional>
#include <tuple>
#include <QObject>

namespace Aurora {
namespace Mdm {

class ManagedConfigurationValuePrivate;
class Q_DECL_EXPORT ManagedConfigurationValue
{
public:
    enum class Type {
        Unknown = -1,
        Integer,
        String,
        Bool,
        IntegerArray,
        StringArray,
        BoolArray,
        Select,
        MultiSelect,
        Structure,
    };

    enum class ConfigurationType {
        Unknown = -1,
        Admin,
        User,
        Both,
    };

    enum class Reason {
        Unknown = -1,
        Unset,
        SetByDefault,
        SetByAdmin,
        SetByUser,
    };

    enum class Detail {
        Unknown = -1,
        Title,
        Description,
        Type,
        Value,
        DefaultValue,
        ConfigurationType,
        Reason,
        Options,
    };

    struct Option
    {
        QString value;
        QString description;
    };

    ManagedConfigurationValue();
    ManagedConfigurationValue(Type type, QVariant value);
    ManagedConfigurationValue(const ManagedConfigurationValue &other);
    ManagedConfigurationValue(ManagedConfigurationValue &&other);
    virtual ~ManagedConfigurationValue();

    ManagedConfigurationValue &operator=(const ManagedConfigurationValue &other);
    ManagedConfigurationValue &operator=(ManagedConfigurationValue &&other);

    Type type() const;
    QVariant value() const;
    bool set(Type type, const QVariant &value);
    bool isValid() const;

    QString title() const;
    QString description() const;
    QVariant getDetail(Detail detail) const;

private:
    ManagedConfigurationValue(ManagedConfigurationValuePrivate *d);

    Q_DECLARE_PRIVATE(ManagedConfigurationValue);
    QScopedPointer<ManagedConfigurationValuePrivate> d_ptr;
};

class ManagedConfigurationPrivate;
class Q_DECL_EXPORT ManagedConfiguration
{
public:
    ManagedConfiguration();
    ManagedConfiguration(const ManagedConfiguration &other);
    ManagedConfiguration(ManagedConfiguration &&other);
    virtual ~ManagedConfiguration();

    ManagedConfiguration &operator=(const ManagedConfiguration &other);
    ManagedConfiguration &operator=(ManagedConfiguration &&other);

    QStringList keys() const;
    ManagedConfigurationValue get(const QString &key) const;
    void add(const QString &key, const ManagedConfigurationValue &value);

private:
    Q_DECLARE_PRIVATE(ManagedConfiguration);
    QScopedPointer<ManagedConfigurationPrivate> d_ptr;
};

class ManagedConfigurationManagerPrivate;
class Q_DECL_EXPORT ManagedConfigurationManager : public QObject
{
    Q_OBJECT

public:
    class Error
    {
    public:
        enum Code {
            NoError = 0,
            Internal,
            InvalidType,
            InvalidValue,
            Forbidden,
            NotConfigurable,
            InvalidUserId,
            UnsupportedType,
        };

        Error(Code code = Code::NoError, const QString &message = QString());
        virtual ~Error();

        bool isError() const;
        operator bool() const;

        Code code() const;
        QString message() const;

    private:
        Code m_code;
        QString m_message;
    };

    using GetConfigurationCallback = std::function<void(const ManagedConfiguration &, const Error &)>;
    using ChangeConfigurationCallback = std::function<void(const Error &)>;

    explicit ManagedConfigurationManager(QObject *parent = nullptr);
    virtual ~ManagedConfigurationManager();

    std::tuple<QStringList, Error> getApplications();
    void getConfiguration(const QString &applicationId,
                          int userId,
                          const GetConfigurationCallback &callback);
    void changeConfiguration(const QString &applicationId,
                             int userId,
                             const ManagedConfiguration &configuration,
                             const ChangeConfigurationCallback &callback);

private:
    Q_DISABLE_COPY(ManagedConfigurationManager)
    Q_DECLARE_PRIVATE(ManagedConfigurationManager)
    QScopedPointer<ManagedConfigurationManagerPrivate> d_ptr;
};

} // namespace Mdm
} // namespace Aurora

Q_DECLARE_METATYPE(Aurora::Mdm::ManagedConfigurationValue);
Q_DECLARE_METATYPE(Aurora::Mdm::ManagedConfigurationValue::ConfigurationType);
Q_DECLARE_METATYPE(Aurora::Mdm::ManagedConfigurationValue::Reason);
Q_DECLARE_METATYPE(Aurora::Mdm::ManagedConfigurationValue::Type);
Q_DECLARE_METATYPE(Aurora::Mdm::ManagedConfigurationValue::Detail);
Q_DECLARE_METATYPE(Aurora::Mdm::ManagedConfigurationValue::Option);
Q_DECLARE_METATYPE(Aurora::Mdm::ManagedConfiguration);

#endif // AURORA_MDM_MANAGEDCONFIGURATION_H
